## 4a: Alternate method for estimating drawdowns in other assets
## Cross-sectional HILDA data across age groups (rather than balanced longitudinal panel data) used to 
## estimate difference between observed change in wealth and estimated change (given assumed rates of return and saving). 
## Results in similar patterns of drawdowns but some higher rates, particularly for older homeowners.

# Prelims -----------------------------------------------------------------

gc()

## custom smoother function
source("./R scripts/Data and parameters/0 Custom functions.R")



# Read in data on rates of return and saving rates ------------------------

other_assets_return_rates <- qread("./Input data/other_asset_returns_a.qs") %>% 
  select(-other_assets_return_rate, other_assets_return_rate=other_assets_return_rate_smooth)

other_assets_saving_rates <- qread("./Input data/saving_other_aih.qs") %>% 
  select(-age_grp_hes)


# Read in HILDA grouped master data ------------------------------------------------------------

hilda_grouped_master <- qread("./Input data/Intermediate input data/hilda_grouped_master.qs") 

wealth_years <- c(2, 6, 10, 14, 18)

hilda_grouped <- hilda_grouped_master %>% 
  filter(wavenumber %in% wealth_years) %>% 
  mutate(homeowner = ifelse(housing_assets>0, 1, 0))


# Estimate drawdowns ------------------------------------------------------

## Summarise data by vars for joining saving and return rates
implied_drawdown <- hilda_grouped %>%
  mutate(total_inc_qtile = as.numeric(total_inc_qtile)) %>% ## for later merging
  group_by(wavenumber, age_grp, total_inc_qtile, homeowner) %>% 
  summarise(n = sum(hhwte),
            av_inc = wtd.mean(total_inc, weights=hhwte), ## inc that is saved
            av_other_assets = wtd.mean(other_assets, weights=hhwte)
            ) %>% 
  left_join(other_assets_return_rates) %>% 
  left_join(other_assets_saving_rates) %>% 
  ## remove inc grouping
  group_by(wavenumber, age_grp, homeowner) %>% 
  summarise(across(c(av_inc, av_other_assets, other_assets_return_rate, saving_rate_other),
                   ~wtd.mean(.x, weights=n)),
            n=sum(n)) %>% 
  ## if we assume that ppl in each group for 5 years earn the same income increasing by 2.5% and save it at the estimated rate, and they earn returns on other assets at given rates...
  ## what is the implied value of other assets at the end?
  mutate(implied_future_other_assets = 
           ifelse(av_inc>0, 
                  av_other_assets*(1+other_assets_return_rate)^5 +
                    av_inc*1.025 *saving_rate_other*(1+other_assets_return_rate)^4 +
                    av_inc*1.025^2 *saving_rate_other*(1+other_assets_return_rate)^3 +
                    av_inc*1.025^3 *saving_rate_other*(1+other_assets_return_rate)^2 +
                    av_inc*1.025^4 *saving_rate_other*(1+other_assets_return_rate) +
                    av_inc*1.025^5 *saving_rate_other,
                  av_other_assets*(1+other_assets_return_rate)^5)
         ) %>% 
  group_by(wavenumber, homeowner) %>% 
  ## difference between implied and actual at the next age group up
  arrange(wavenumber, homeowner, age_grp) %>% 
  mutate(other_assets_diff = implied_future_other_assets - lead(av_other_assets),
         implied_drawdown = other_assets_diff/implied_future_other_assets ) %>% 
  group_by(age_grp, homeowner) %>% 
  summarise(implied_drawdown = mean(implied_drawdown, na.rm=T)/5,
            av_other_assets = wtd.mean(av_other_assets, weights=n)) %>%  ## divide by 5 ~ over 5 years
  ## positive means drawdown. negative means they have more other assets than implied
  ## negative vals. Force drawdowns to be zero for them. and for anyone <15,20. For 85-105 non homeowners, make rate same as 80-85s
  mutate(implied_drawdown = ifelse(implied_drawdown<0 | age_grp<="[15,20)", 0, implied_drawdown),
         age85 = ifelse(age_grp>="[85,90)", "[85,105]", as.character(age_grp) )) %>% 
  group_by(age85, homeowner) %>% 
  mutate(implied_drawdown = mean(implied_drawdown, na.rm=T)) %>% 
  group_by(homeowner) %>% 
  mutate(implied_drawdown_smooth = custom_smoother(implied_drawdown) ) 
## note the higher implied drawdowns here for the old could be because of overestimating future inc when old (whereas other approach used past actual inc)
  
  
ggplot(implied_drawdown ) +
  geom_col(aes(x=age_grp, y=implied_drawdown_smooth, fill=as.factor(homeowner)), position="dodge")


# Starting cohort data ----------------------------------------------------

starting_cohorts_pathways <- qread(starting_cohorts_pathways_file)



# Put into new version of starting_cohorts, convert to year list and save --------

## make homeowner variable align with cohorts-pathways which includes -1
implied_drawdown_adj <- expand_grid(age_grp=unique(starting_cohorts_pathways$age_grp), ho=unique(starting_cohorts_pathways$ho)) %>% 
  left_join(implied_drawdown %>% select(age_grp, ho=homeowner, implied_drawdown_smooth)) %>% 
  ## create new rows for homeowner==-1 ie bought a house this year in model (same as homeowner==1 ie bought a house previously)
  left_join(implied_drawdown %>% 
              filter(homeowner==1) %>% 
              mutate(homeowner=-1) %>% 
              select(age_grp, ho=homeowner, implied_drawdown_smooth2=implied_drawdown_smooth)) %>% 
  mutate(implied_drawdown_smooth = ifelse(is.na(implied_drawdown_smooth), implied_drawdown_smooth2, implied_drawdown_smooth)) %>% 
  select(-implied_drawdown_smooth2)

## attach to starting_cohorts_pathways
starting_cohorts_pathways_adj <- starting_cohorts_pathways %>% 
  left_join(implied_drawdown_adj ) %>% 
  mutate(other_asset_drawdown_param=implied_drawdown_smooth) %>% 
  select(-implied_drawdown_smooth)



## split into list by year - save for use in model running
year_list_0 <- starting_cohorts_pathways_adj %>% 
  split(., .$year) 

qsave(year_list_0, "./Output data/year_list_0_alt_other_drawdown.qs")

